##############################################
### Libraries required
##############################################

library(nsp)
library(stepR)
library(strucchange)

##############################################
### Loading the data needed
##############################################

load("nsp.Rdata")

# This loads the following datasets:
#
# Signals for simulation studies of Section 5:
#
# hat.sig
# wide.teeth
# teeth10
# blocks
# wave2sect
# squarewave
#
# Real data for Sections 6.1 and 6.2:
#
# real_dat
# real_dat_sc
# nmp



##############################################
### Function definitions
##############################################

bp_coverage_fromdata <- function(truth, data, breaks = NULL, alpha = 0.1) {
	
	est <- breakpoints(data ~ 1, h = 2, breaks = breaks)
	
	if (is.na(est$breakpoints[1])) {

		res = rep(TRUE, 0)		
		coverage = TRUE
		prop = NaN
		all.intervals = 0
		good.intervals = 0
		avg.length = NaN
		
	} else {
		
		b <- length(est$breakpoints)
		conf <- confint(est, level = 1 - alpha / b)
		d <- dim(conf$confint)
		res <- rep(TRUE, d[1])
		
		cpts <- which(abs(diff(truth)) > 0)

		for (i in 1:d[1]) {
			
			cur_int <- (1+conf$confint[i,1]):(conf$confint[i,3]-1)
			res[i] <- any(cpts %in% cur_int)
			
		}
		
		coverage <- all(res)
		prop <- mean(res)
		good.intervals <- sum(res)
		all.intervals <- d[1]
		avg.length <- mean(conf$confint[res,3] - conf$confint[res,1])
		
	}
	
	list(res = res, coverage = coverage, prop = prop, good.intervals = good.intervals, all.intervals = all.intervals, avg.length = avg.length)	
	
}


smuce_coverage_fromdata <- function(truth, data, alpha = 0.1) {
		
	est <- stepFit(data, alpha = alpha, confband=TRUE)
	
	j <- jumpint(est)
	d <- dim(j)
	
	res <- rep(TRUE, d[1]-1)
	
	cpts <- which(abs(diff(truth)) > 0)
		
	if (d[1]-1) for (i in 1:(d[1]-1)) {
		
		cur_int <- j[i,3]:j[i,4]
		res[i] <- any(cpts %in% cur_int)
	}

	coverage <- all(res)
	prop <- mean(res)
	good.intervals <- sum(res)
	all.intervals <- d[1] - 1
	avg.length <- mean(j[c(res, FALSE),4]-j[c(res, FALSE),3]+2)

	list(res = res, coverage = coverage, prop = prop, good.intervals = good.intervals, all.intervals = all.intervals, avg.length = avg.length)	

}


nsp_coverage_fromdata <- function(truth, data, thresh.type, overlap, alpha = 0.1, sigma = NULL) {
	
	est <- nsp_poly(data, thresh.type = thresh.type, sigma = sigma, alpha = alpha, overlap = overlap)
	
	d <- dim(est$intervals)

	res <- rep(TRUE, d[1])

	cpts <- which(abs(diff(truth)) > 0)
	
	if (d[1]) for (i in 1:(d[1])) {
		
		cur_int <- est$intervals[i,1]:(est$intervals[i,2]-1)
		res[i] <- any(cpts %in% cur_int)
		
	}

	coverage <- all(res)
	prop <- mean(res)
	good.intervals <- sum(res)
	all.intervals <- d[1]
	avg.length <- mean(est$intervals[res,2] - est$intervals[res,1] + 1)

	list(res = res, coverage = coverage, prop = prop, good.intervals = good.intervals, all.intervals = all.intervals, avg.length = avg.length)	

	
}


nsp_sn_coverage_fromdata <- function(truth, data, overlap, alpha = 0.1) {
	
	est <- nsp_poly_selfnorm(data, alpha = alpha, overlap = overlap)
	
	d <- dim(est$intervals)

	res <- rep(TRUE, d[1])

	cpts <- which(abs(diff(truth)) > 0)
	
	if (d[1]) for (i in 1:(d[1])) {
		
		cur_int <- est$intervals[i,1]:(est$intervals[i,2]-1)
		res[i] <- any(cpts %in% cur_int)
		
	}

	coverage <- all(res)
	prop <- mean(res)
	good.intervals <- sum(res)
	all.intervals <- d[1]
	avg.length <- mean(est$intervals[res,2] - est$intervals[res,1] + 1)

	list(res = res, coverage = coverage, prop = prop, good.intervals = good.intervals, all.intervals = all.intervals, avg.length = avg.length)	
	
}


unlist_tail <- function(llist, how.many = 5) {
	
	d <- unlist(llist)
	n <- length(d)
	as.numeric(d[(n-how.many+1):n])
	
}


avg.omit.nan <- function(A) {
	
	d <- dim(A)
	
	res <- rep(0, d[1])
	for (i in 1:d[1]) {
		x <- A[i,]
		res[i] <- mean(x[!is.nan(x)])
	}
	
	res	
	
}


plain.100 <- function() {
	
	list(model = rep(0, 100), path = rnorm(100))
	
}


plain.300 <- function() {
	
	list(model = rep(0, 300), path = rnorm(300))
	
}


sig.100.1 <- function() {
	
	f <- c(rep(0, 50), rep(1, 50))
	list(model = f, path = f + rnorm(100))
	
}


sig.300.1 <- function() {

	f <- c(rep(0, 150), rep(1, 150))
	list(model = f, path = f + rnorm(300))
		
}


hat <- function() {
	
	f <- hat.sig
	list(model = f, path = f + 100 * rnorm(400))
	
}


wideteeth <- function() {
	
	f <- wide.teeth
	list(model = f, path = f + rnorm(300))
	
}


teeth.10 <- function() {
	
	f <- teeth10
	list(model = f, path = f + rnorm(140) * .4)
		
}


blox <- function() {
	
	f <- blocks
	list(model = f, path = f + 10 * rnorm(2048))
	
}


plain.300.01 <- function() {
	list(model = rep(0, 300), path = (1 - 0.1^2)^(1/2) * arima.sim(list(ar = 0.1), n = 300))
}


plain.300.03 <- function() {
	list(model = rep(0, 300), path = (1 - 0.3^2)^(1/2) * arima.sim(list(ar = 0.3), n = 300))
}


plain.300.05 <- function() {
	list(model = rep(0, 300), path = (1 - 0.5^2)^(1/2) * arima.sim(list(ar = 0.5), n = 300))
}


plain.300.07 <- function() {
	list(model = rep(0, 300), path = (1 - 0.7^2)^(1/2) * arima.sim(list(ar = 0.7), n = 300))
}


sig.300.1.01 <- function() {
	f <- c(rep(0, 150), rep(1, 150))
	list(model = f, path = f + (1 - 0.1^2)^(1/2) * arima.sim(list(ar = 0.1), n = 300))
}


sig.300.1.03 <- function() {
	f <- c(rep(0, 150), rep(1, 150))
	list(model = f, path = f + (1 - 0.3^2)^(1/2) * arima.sim(list(ar = 0.3), n = 300))
}


sig.300.1.05 <- function() {
	f <- c(rep(0, 150), rep(1, 150))
	list(model = f, path = f + (1 - 0.5^2)^(1/2) * arima.sim(list(ar = 0.5), n = 300))
}


sig.300.1.07 <- function() {
	f <- c(rep(0, 150), rep(1, 150))
	list(model = f, path = f + (1 - 0.7^2)^(1/2) * arima.sim(list(ar = 0.7), n = 300))
}


rts <- function(n, k) {
	
	rt(n, k) * sqrt((k-2) / k)
	
}


plain.300.t.3 <- function() {
	list(model = rep(0, 300), path = rts(300, 3))
}


sig.300.t.3 <- function() {
	f <- c(rep(0, 150), rep(1, 150))
	list(model = f, path = f + rts(300, 3))
}


inner.prod.iter <- function(x) {
		
	m <- length(x)
	z <- cumsum(x)

	ip <- sqrt(((m-1):1) / m / (1:(m-1))) * z[1:(m-1)] - sqrt((1:(m-1)) / m / ((m-1):1)) * (z[m] - z[1:(m-1)])
	
	abs(ip)	
}


mean.from.cpt <- function(x, cpt) {
  n <- length(x)
  len.cpt <- length(cpt)
  if (len.cpt) cpt <- sort(cpt)
  beg <- endd <- rep(0, len.cpt+1)
  beg[1] <- 1
  endd[len.cpt+1] <- n
  if (len.cpt) {
    beg[2:(len.cpt+1)] <- cpt+1
    endd[1:len.cpt] <- cpt
  }
  means <- rep(0, len.cpt+1)
  for (i in 1:(len.cpt+1)) means[i] <- mean(x[beg[i]:endd[i]])
  rep(means, endd-beg+1)
}



#####################################################
### Section 5.1 -- code
#####################################################



sim_cpt_coverage_1 <- function(simulator, N = 100, seed = 1, sigma = NULL) {
	
	res.nsp.univ.no <- res.nsp.sim.no <- res.nsp.univ.o <- res.nsp.sim.o <- res.bp.nolimit <- res.bp.limit <- res.smuce <- res.nsp.sn <- matrix(0, 5, N)	

	set.seed(seed)
	
	for (i in 1:N) {
		print(i)
		x <- simulator()	

		a <- nsp_coverage_fromdata(x$model, x$path, "univ", FALSE, sigma = sigma)
		res.nsp.univ.no[,i] <- unlist_tail(a)
		
		a <- nsp_coverage_fromdata(x$model, x$path, "sim", FALSE, sigma = sigma)
		res.nsp.sim.no[,i] <- unlist_tail(a)

		a <- nsp_coverage_fromdata(x$model, x$path, "univ", TRUE, sigma = sigma)
		res.nsp.univ.o[,i] <- unlist_tail(a)

		a <- nsp_coverage_fromdata(x$model, x$path, "sim", TRUE, sigma = sigma)
		res.nsp.sim.o[,i] <- unlist_tail(a)
		
#		a <- nsp_sn_coverage_fromdata(x$model, x$path, FALSE)
#		res.nsp.sn[,i] <- unlist_tail(a)
		
		a <- bp_coverage_fromdata(x$model, x$path)
		res.bp.nolimit[,i] <- unlist_tail(a)
		
		a <- bp_coverage_fromdata(x$model, x$path, breaks = max(1, res.nsp.univ.no[4,i]))
		res.bp.limit[,i] <- unlist_tail(a)

		a <- smuce_coverage_fromdata(x$model, x$path)
		res.smuce[,i] <- unlist_tail(a)
		
	}		

	list(res.nsp.univ.no = res.nsp.univ.no, res.nsp.sim.no = res.nsp.sim.no,
			res.nsp.univ.o = res.nsp.univ.o, res.nsp.sim.o = res.nsp.sim.o, res.nsp.sn = res.nsp.sn,
			res.bp.nolimit = res.bp.nolimit, res.bp.limit = res.bp.limit,
			res.smuce = res.smuce)		
	
}


plain.100.sim <- sim_cpt_coverage_1(plain.100)

# To obtain the figures for the 1st line [Noise 100] of Table 2, do

lapply(plain.100.sim, avg.omit.nan)

# and take the first component of each vector returned.



plain.300.sim <- sim_cpt_coverage_1(plain.300)

# To obtain the figures for the 2nd line [Noise 300] of Table 2, do

lapply(plain.300.sim, avg.omit.nan)

# and take the first component of each vector returned.



sig.100.1.sim <- sim_cpt_coverage_1(sig.100.1)

# To obtain the "Single 100" entries for Table 3, do

lapply(sig.100.1.sim, avg.omit.nan)



sig.300.1.sim <- sim_cpt_coverage_1(sig.300.1)

# To obtain the "Single 300" entries for Table 3, do

lapply(sig.300.1.sim, avg.omit.nan)



hat.sim <- sim_cpt_coverage_1(hat)

# To obtain the "Wave" entries for Table 3, do

lapply(hat.sim, avg.omit.nan)



wideteeth.sim <- sim_cpt_coverage_1(wideteeth)

# To obtain the "Wide Teeth" entries for Table 3, do

lapply(wideteeth.sim, avg.omit.nan)



teeth.10.sim <- sim_cpt_coverage_1(teeth.10, N = 50)

# To obtain the "Teeth 10" entries for Table 3, do

lapply(teeth.10.sim, avg.omit.nan)



sim_cpt_coverage_2 <- function(simulator, N = 100, seed = 1, sigma = NULL) {
	
	res.nsp.univ.no <- res.nsp.sim.no <- res.nsp.univ.o <- res.nsp.sim.o <- res.bp.nolimit <- res.bp.limit <- res.smuce <- res.nsp.sn <- matrix(0, 5, N)	

	set.seed(seed)
	
	for (i in 1:N) {
		print(i)
		x <- simulator()	

		a <- nsp_coverage_fromdata(x$model, x$path, "univ", FALSE, sigma = sigma)
		res.nsp.univ.no[,i] <- unlist_tail(a)
		
		a <- nsp_coverage_fromdata(x$model, x$path, "sim", FALSE, sigma = sigma)
		res.nsp.sim.no[,i] <- unlist_tail(a)

		a <- nsp_coverage_fromdata(x$model, x$path, "univ", TRUE, sigma = sigma)
		res.nsp.univ.o[,i] <- unlist_tail(a)

		a <- nsp_coverage_fromdata(x$model, x$path, "sim", TRUE, sigma = sigma)
		res.nsp.sim.o[,i] <- unlist_tail(a)
		
#		a <- nsp_sn_coverage_fromdata(x$model, x$path, FALSE)
#		res.nsp.sn[,i] <- unlist_tail(a)
		
#		a <- bp_coverage_fromdata(x$model, x$path)
#		res.bp.nolimit[,i] <- unlist_tail(a)
		
#		a <- bp_coverage_fromdata(x$model, x$path, breaks = max(1, res.nsp.univ.no[4,i]))
#		res.bp.limit[,i] <- unlist_tail(a)

		a <- smuce_coverage_fromdata(x$model, x$path)
		res.smuce[,i] <- unlist_tail(a)
		
	}		

	list(res.nsp.univ.no = res.nsp.univ.no, res.nsp.sim.no = res.nsp.sim.no,
			res.nsp.univ.o = res.nsp.univ.o, res.nsp.sim.o = res.nsp.sim.o, res.nsp.sn = res.nsp.sn,
			res.bp.nolimit = res.bp.nolimit, res.bp.limit = res.bp.limit,
			res.smuce = res.smuce)		
	
}



blocks.sim <- sim_cpt_coverage_2(blox, N = 100)

# To obtain the "Blocks" entries for Table 2, do

lapply(blocks.sim, avg.omit.nan)




sim_cpt_coverage_3 <- function(simulator, N = 100, seed = 1, sigma = NULL) {
	
	res.nsp.univ.no <- res.nsp.sim.no <- res.nsp.univ.o <- res.nsp.sim.o <- res.bp.nolimit <- res.bp.limit <- res.smuce <- res.nsp.sn <- matrix(0, 5, N)	

	set.seed(seed)
	
	for (i in 1:N) {
		print(i)
		x <- simulator()	

		a <- nsp_coverage_fromdata(x$model, x$path, "univ", FALSE, sigma = sigma)
		res.nsp.univ.no[,i] <- unlist_tail(a)
		
		a <- nsp_coverage_fromdata(x$model, x$path, "sim", FALSE, sigma = sigma)
		res.nsp.sim.no[,i] <- unlist_tail(a)

		a <- nsp_coverage_fromdata(x$model, x$path, "univ", TRUE, sigma = sigma)
		res.nsp.univ.o[,i] <- unlist_tail(a)

		a <- nsp_coverage_fromdata(x$model, x$path, "sim", TRUE, sigma = sigma)
		res.nsp.sim.o[,i] <- unlist_tail(a)
		
#		a <- nsp_sn_coverage_fromdata(x$model, x$path, FALSE)
#		res.nsp.sn[,i] <- unlist_tail(a)
		
#		a <- bp_coverage_fromdata(x$model, x$path)
#		res.bp.nolimit[,i] <- unlist_tail(a)
		
#		a <- bp_coverage_fromdata(x$model, x$path, breaks = max(1, res.nsp.univ.no[4,i]))
#		res.bp.limit[,i] <- unlist_tail(a)

#		a <- smuce_coverage_fromdata(x$model, x$path)
#		res.smuce[,i] <- unlist_tail(a)
		
	}		

	list(res.nsp.univ.no = res.nsp.univ.no, res.nsp.sim.no = res.nsp.sim.no,
			res.nsp.univ.o = res.nsp.univ.o, res.nsp.sim.o = res.nsp.sim.o, res.nsp.sn = res.nsp.sn,
			res.bp.nolimit = res.bp.nolimit, res.bp.limit = res.bp.limit,
			res.smuce = res.smuce)		
	
}


plain.300.01.lrv.sim <- sim_cpt_coverage_3(plain.300.01, N = 100, sigma = sqrt((1+.1)/(1-.1)))
plain.300.03.lrv.sim <- sim_cpt_coverage_3(plain.300.03, N = 100, sigma = sqrt((1+.3)/(1-.3)))
plain.300.05.lrv.sim <- sim_cpt_coverage_3(plain.300.05, N = 100, sigma = sqrt((1+.5)/(1-.5)))
plain.300.07.lrv.sim <- sim_cpt_coverage_3(plain.300.07, N = 100, sigma = sqrt((1+.7)/(1-.7)))


sig.300.1.01.lrv.sim <- sim_cpt_coverage_3(sig.300.1.01, N = 100, sigma = sqrt((1+.1)/(1-.1)))
sig.300.1.03.lrv.sim <- sim_cpt_coverage_3(sig.300.1.03, N = 100, sigma = sqrt((1+.3)/(1-.3)))
sig.300.1.05.lrv.sim <- sim_cpt_coverage_3(sig.300.1.05, N = 100, sigma = sqrt((1+.5)/(1-.5)))
sig.300.1.07.lrv.sim <- sim_cpt_coverage_3(sig.300.1.07, N = 100, sigma = sqrt((1+.7)/(1-.7)))

# Use e.g.

lapply(plain.300.01.lrv.sim, avg.omit.nan)

# etc. to obtain the entries in Tables 4 and 5.




# Blocks example from Section 5.1

set.seed(1)
x <- blocks + 10 * rnorm(2048)

x.n <- nsp_poly(x)
ts.plot(x, col="grey", ylab="")
draw_rects(x.n, c(-40, 50), 20, "red")
truecpt <- which(abs(diff(blocks)) > 0)
abline(v = truecpt, col="blue")

x.n.o <- nsp_poly(x, thresh.type = "sim", overlap = TRUE)
ts.plot(x, col="grey", ylab="")
draw_rects(x.n.o, c(-40, 50), 20, "red")
abline(v = truecpt, col="blue")

cpt_importance(x.n)


#####################################################
### Section 5.2 -- code
#####################################################


set.seed(1)
xw <- wave2sect + rnorm(450)/2
xw.n.0 <- nsp_poly(xw)
xw.n.1 <- nsp_poly(xw, deg = 1)
xw.n.2 <- nsp_poly(xw, deg = 2)

ts.plot(xw, col="grey", ylab="")
lines(wave2sect)
draw_rects(xw.n.0, c(-1, 5), 20, "brown")


ts.plot(xw, col="grey", ylab="")
lines(wave2sect)
draw_rects(xw.n.1, c(-1, 5), 20, "red")


ts.plot(xw, col="grey", ylab="")
lines(wave2sect)
draw_rects(xw.n.2, c(-1, 5), 20, "brown")


#####################################################
### Section 5.3 -- code
#####################################################

set.seed(1)
x.rt.hard <- squarewave + rt(800, 4) * seq(from = 2, to = 8, length = 800)
x.rt.hard.sn <- nsp_poly_selfnorm(x.rt.hard)
ts.plot(x.rt.hard, ylab="")
draw_rects(x.rt.hard.sn, c(-70, 60), 20, "red")
abline(v = c(200, 400, 600), col="blue")


sim_cpt_coverage_4 <- function(simulator, N = 100, seed = 1, sigma = NULL) {
	
	res.nsp.univ.no <- res.nsp.sim.no <- res.nsp.univ.o <- res.nsp.sim.o <- res.bp.nolimit <- res.bp.limit <- res.smuce <- res.nsp.sn <- matrix(0, 5, N)	

	set.seed(seed)
	
	for (i in 1:N) {
		print(i)
		x <- simulator()	

#		a <- nsp_coverage_fromdata(x$model, x$path, "univ", FALSE, sigma = sigma)
#		res.nsp.univ.no[,i] <- unlist_tail(a)
		
#		a <- nsp_coverage_fromdata(x$model, x$path, "sim", FALSE, sigma = sigma)
#		res.nsp.sim.no[,i] <- unlist_tail(a)

#		a <- nsp_coverage_fromdata(x$model, x$path, "univ", TRUE, sigma = sigma)
#		res.nsp.univ.o[,i] <- unlist_tail(a)

#		a <- nsp_coverage_fromdata(x$model, x$path, "sim", TRUE, sigma = sigma)
#		res.nsp.sim.o[,i] <- unlist_tail(a)
		
		a <- nsp_sn_coverage_fromdata(x$model, x$path, FALSE)
		res.nsp.sn[,i] <- unlist_tail(a)
		
#		a <- bp_coverage_fromdata(x$model, x$path)
#		res.bp.nolimit[,i] <- unlist_tail(a)
		
#		a <- bp_coverage_fromdata(x$model, x$path, breaks = max(1, res.nsp.univ.no[4,i]))
#		res.bp.limit[,i] <- unlist_tail(a)

#		a <- smuce_coverage_fromdata(x$model, x$path)
#		res.smuce[,i] <- unlist_tail(a)
		
	}		

	list(res.nsp.univ.no = res.nsp.univ.no, res.nsp.sim.no = res.nsp.sim.no,
			res.nsp.univ.o = res.nsp.univ.o, res.nsp.sim.o = res.nsp.sim.o, res.nsp.sn = res.nsp.sn,
			res.bp.nolimit = res.bp.nolimit, res.bp.limit = res.bp.limit,
			res.smuce = res.smuce)		
	
}



plain.300.t.3.sim <- sim_cpt_coverage_4(plain.300.t.3, N = 100)
sig.300.t.3.sim <- sim_cpt_coverage_4(sig.300.t.3, N = 100)


# To obtain the numbers for the final paragraph of Section 5.3, do

lapply(plain.300.t.3.sim, avg.omit.nan)

lapply(sig.300.t.3.sim, avg.omit.nan)


#####################################################
### Section 6.1 -- code
#####################################################

# real_dat is the original (unscaled) time series as described in Section 6.1 of the paper

r.n <- nsp_poly(real_dat)

# This returns
#
#$intervals
#  starts ends   values midpoints
#1     24   55 7.320196        39
#2     76   83 8.740810        79
#
#$threshold.used
#[1] 7.102313

# Estimating change-point locations via CUSUM within each interval:

which.max(abs(inner.prod.iter(real_dat[24:55]))) + 23

#[1] 47

which.max(abs(inner.prod.iter(real_dat[76:83]))) + 75

#[1] 82

sig_const_orig <- mean.from.cpt(real_dat, c(47, 82))
ts.plot(real_dat, xlab="Time (quarters)", ylab="")
lines(sig_const_orig, col="red", lwd = 3)
draw_rects(r.n, c(-10, 15), col="grey", density = 20)

# real_dat_sc is real_dat scaled interval-wise on 1:47, 48:82, 83:103 by the corresponding sample standard
# deviations of real_dat, as described in the paper


r.s.n <- nsp_poly(real_dat_sc)
r.s.n
#$intervals
#  starts ends   values midpoints
#1     23   54 3.505574        38
#2     76   84 3.460878        80
#
#$threshold.used
#[1] 3.442165


which.max(abs(inner.prod.iter(real_dat_sc[76:84]))) + 75

#[1] 82

which.max(abs(inner.prod.iter(real_dat_sc[23:54]))) + 22

#[1] 47


sig_const <- mean.from.cpt(real_dat_sc, c(47, 82))
ts.plot(real_dat_sc, xlab="Time (quarters)", ylab="")
lines(sig_const, col="red", lwd = 3)
draw_rects(r.s.n, c(-3, 5), col="grey", density = 20)



nsp_poly(real_dat_sc[1:47])

#$intervals
#[1] starts    ends      values    midpoints
#<0 rows> (or 0-length row.names)
#
#$threshold.used
#[1] 3.565766


nsp_poly(real_dat_sc, deg = 1)

#$intervals
#  starts ends  values midpoints
#1     57   84 3.49048        70
#
#$threshold.used
#[1] 3.442165

which.max(abs(inner.prod.iter(real_dat_sc[58:103]))) + 57

#[1] 79



real_dat_sc_1alt <- real_dat_sc[1:79]
time_1alt <- 1:79
sig_1alt <- lm(real_dat_sc_1alt ~ time_1alt)$fitted
sig_2alt <- rep(mean(real_dat_sc[80:103]), 103-80+1)
sig_linalt <- c(sig_1alt, sig_2alt)



# BIC for linear+const

103/2 * log(var(real_dat_sc - sig_linalt)) + 4/2 * log(103)

# BIC for piecewise-constant

103/2 * log(var(real_dat_sc - sig_const)) + 5/2 * log(103)


ts.plot(real_dat_sc, xlab="Time (quarters)", ylab="")
lines(sig_linalt, col="red", lwd = 3)




sig1orig <- real_dat[1:79]
tim <- 1:79
tim2 <- tim^2 / 79
dd <- lm(sig1orig ~ tim + tim2)$fitted
sig_quad <- c(dd, rep(mean(real_dat[80:103]), 103-80+1))

ts.plot(real_dat, xlab="Time (quarters)", ylab="")
lines(sig_quad, col="red", lwd = 3)



sig_const_orig <- mean.from.cpt(real_dat, c(47, 82))

var(real_dat - sig_quad)
#[1] 4.900494

var(real_dat - sig_const_orig)
#[1] 4.939679


#####################################################
### Section 6.2 -- code
#####################################################

# Time series stored in variable nmp.

covv <- matrix(1, 130, 2)

covv[,2] <- log(nmp[1:130])

resp <- log(nmp[2:131])

nsp_tvreg(resp, covv, 1000, overlap=TRUE)

#$intervals
#  starts ends     values midpoints
#1     24   96 0.04095207        60
#
#$threshold.used
#[1] 0.04038574





summary(lm(resp[1:60]~covv[1:60,]-1))

summary(lm(resp[61:130]~covv[61:130,]-1))



ts.plot(log(nmp), ylab="", xlab="Time (months) starting January 2010")

abline(v = 60, col="red")
